/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wildfly.camel.test.weka;

import java.io.InputStream;

import org.apache.camel.CamelContext;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.impl.DefaultCamelContext;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.wildfly.extension.camel.CamelAware;

import io.nessus.weka.Dataset;
import weka.classifiers.Classifier;
import weka.classifiers.Evaluation;

@CamelAware
@RunWith(Arquillian.class)
public class WekaIntegrationTest {
    
    @Deployment
    public static JavaArchive createDeployment() {
        JavaArchive archive = ShrinkWrap.create(JavaArchive.class, "camel-weka-tests");
        archive.addAsResource("weka/sfny-train.arff");
		return archive;
    }

    @Test
    public void testJ48WithCrossValidation() throws Exception {
        
        CamelContext camelctx = new DefaultCamelContext();
            
        camelctx.addRoutes(new RouteBuilder() {
            
            @Override
            public void configure() throws Exception {
                
                    // Client passes in the training data
                    from("direct:start")
                    
                    // Build a J48 classifier using cross-validation with 10 folds
                    .to("weka:model?build=J48&xval=true&folds=10&seed=1")
                    
                    // Persist the J48 model
                    .to("weka:model?saveTo=target/sfny-j48.model");
            }
        });
        camelctx.start();
        
        try  (InputStream stream = WekaIntegrationTest.class.getResourceAsStream("/weka/sfny-train.arff")) {

            ProducerTemplate producer = camelctx.createProducerTemplate();
            Dataset dataset = producer.requestBody("direct:start", stream, Dataset.class);
            
            Classifier classifier = dataset.getClassifier();
            Assert.assertNotNull(classifier);
            
            Evaluation evaluation = dataset.getEvaluation();
            Assert.assertNotNull(evaluation);
        
        } finally {
			camelctx.stop();
		}
    }
}
